<?php
require_once 'database.php';

// Get database connection
$conn = getDBConnection(); // Add this line to establish connection

$logFile = 'received_sessions_log.txt';
function logToFile($message) {
    global $logFile;
    file_put_contents($logFile, "[" . date('Y-m-d H:i:s') . "] $message" . PHP_EOL, FILE_APPEND);
}

// Check DB connection
if ($conn->connect_error) {
    logToFile("Database connection failed: " . $conn->connect_error);
    http_response_code(500);
    exit('Database connection error');
}

// Get and parse JSON payload
$rawInput = file_get_contents('php://input');
$payload = json_decode($rawInput, true);

if (!$payload) {
    logToFile("Bad JSON received: $rawInput");
    http_response_code(400);
    exit('Bad JSON');
}

$user = $payload['user'] ?? '';
$rx   = (int)($payload['rx'] ?? 0);
$tx   = (int)($payload['tx'] ?? 0);

if ($user === '') {
    logToFile("Missing 'user' field in payload: " . json_encode($payload));
    http_response_code(400);
    exit('No user');
}

logToFile("Received USER: $user | RX: $rx | TX: $tx");

// Insert user if not already present
try {
    $stmt = $conn->prepare("INSERT IGNORE INTO users (username, profile_name) VALUES (?, '')");
    if (!$stmt) {
        logToFile("Prepare failed for users INSERT: " . $conn->error);
    } else {
        $stmt->bind_param("s", $user);
        if (!$stmt->execute()) {
            logToFile("Execution failed for users INSERT: " . $stmt->error);
        } else {
            logToFile("User inserted/ignored successfully");
        }
        $stmt->close();
    }
} catch (Exception $e) {
    logToFile("Exception during users insert: " . $e->getMessage());
}

// Insert or update user session
try {
    $stmt = $conn->prepare("
        INSERT INTO user_sessions (username, bytes_rx, bytes_tx, router_time)
        VALUES (?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE 
            bytes_rx = VALUES(bytes_rx),
            bytes_tx = VALUES(bytes_tx),
            router_time = VALUES(router_time)
    ");
    if (!$stmt) {
        logToFile("Prepare failed for user_sessions UPSERT: " . $conn->error);
    } else {
        $stmt->bind_param("sii", $user, $rx, $tx);
        if (!$stmt->execute()) {
            logToFile("Execution failed for user_sessions UPSERT: " . $stmt->error);
        } else {
            logToFile("Session inserted/updated successfully");
        }
        $stmt->close();
    }
} catch (Exception $e) {
    logToFile("Exception during user_sessions UPSERT: " . $e->getMessage());
}

logToFile("Script execution finished successfully");
echo 'OK';
?>
