<?php
session_start();
define('SESSION_TIMEOUT', 900); // 15 minutes.

// Handle logout
if (isset($_GET['logout'])) {
    session_unset();
    session_destroy();
    header("Location: auth-boxed-signin.php");
    exit();
}

if (!isset($_SESSION['phone_number'])) {
    header("Location: auth-boxed-signin.php");
    exit();
}
if (isset($_SESSION['LAST_ACTIVITY']) && (time() - $_SESSION['LAST_ACTIVITY'] > SESSION_TIMEOUT)) {
    session_unset();
    session_destroy();
    header("Location: auth-boxed-signin.php?timeout=1");
    exit();
}
$_SESSION['LAST_ACTIVITY'] = time();

require_once 'database.php';

$conn = getDBConnection();

// --- UNIT FILTER LOGIC ---
$selectedUnit = $_GET['unit'] ?? 'All Units';
$routerFilter = '';
$routerParam = '';
if ($selectedUnit === 'CCTU' || $selectedUnit === 'Accra') {
    $routerFilter = " AND router = ? ";
    $routerParam = $selectedUnit;
}

// Helper function
function getRevenue($conn, $start, $end, $routerFilter = '', $routerParam = '') {
    $sql = "SELECT SUM(amount) as total FROM transaction_history WHERE status='Paid' AND created_at >= ? AND created_at < ? $routerFilter";
    $stmt = $conn->prepare($sql);
    if ($routerFilter) {
        $stmt->bind_param("sss", $start, $end, $routerParam);
    } else {
        $stmt->bind_param("ss", $start, $end);
    }
    $stmt->execute();
    $stmt->bind_result($total);
    $stmt->fetch();
    $stmt->close();
    return floatval($total);
}

// Dates
$today = date('Y-m-d');
$yesterday = date('Y-m-d', strtotime('-1 day'));
$thisWeekStart = date('Y-m-d', strtotime('monday this week'));
$lastWeekStart = date('Y-m-d', strtotime('monday last week'));
$lastWeekEnd = date('Y-m-d', strtotime('sunday last week +1 day'));
$thisMonthStart = date('Y-m-01');
$lastMonthStart = date('Y-m-01', strtotime('first day of last month'));
$lastMonthEnd = date('Y-m-01');
$thisYearStart = date('Y-01-01');
$lastYearStart = date('Y-01-01', strtotime('last year'));
$lastYearEnd = date('Y-01-01');

// Revenue calculations
$revenue_today = getRevenue($conn, $today, date('Y-m-d', strtotime('+1 day')), $routerFilter, $routerParam);
$revenue_yesterday = getRevenue($conn, $yesterday, $today, $routerFilter, $routerParam);

$revenue_this_week = getRevenue($conn, $thisWeekStart, date('Y-m-d', strtotime($thisWeekStart . ' +7 days')), $routerFilter, $routerParam);
$revenue_last_week = getRevenue($conn, $lastWeekStart, $lastWeekEnd, $routerFilter, $routerParam);

$revenue_this_month = getRevenue($conn, $thisMonthStart, date('Y-m-01', strtotime('+1 month')), $routerFilter, $routerParam);
$revenue_last_month = getRevenue($conn, $lastMonthStart, $lastMonthEnd, $routerFilter, $routerParam);

$revenue_this_year = getRevenue($conn, $thisYearStart, date('Y-01-01', strtotime('+1 year')), $routerFilter, $routerParam);
$revenue_last_year = getRevenue($conn, $lastYearStart, $lastYearEnd, $routerFilter, $routerParam);

// Helper for percentage growth
function percentGrowth($current, $previous) {
    if ($previous == 0) return $current > 0 ? 100 : 0;
    return round((($current - $previous) / $previous) * 100, 2);
}

// Growth calculations
$growth_today = percentGrowth($revenue_today, $revenue_yesterday);
$growth_week = percentGrowth($revenue_this_week, $revenue_last_week);
$growth_month = percentGrowth($revenue_this_month, $revenue_last_month);
$growth_year = percentGrowth($revenue_this_year, $revenue_last_year);

// Date ranges
$thisMonthStart = date('Y-m-01');
$nextMonthStart = date('Y-m-01', strtotime('+1 month'));
$lastMonthStart = date('Y-m-01', strtotime('first day of last month'));
$thisMonthEnd = $nextMonthStart;
$lastMonthEnd = $thisMonthStart;

// Helper to get top/bottom profiles
function getProfileSales($conn, $start, $end, $order = 'DESC', $limit = 3, $routerFilter = '', $routerParam = '') {
    $sql = "SELECT profile, SUM(amount) as total, COUNT(*) as count
            FROM transaction_history
            WHERE status='Paid' AND created_at >= ? AND created_at < ? $routerFilter
            GROUP BY profile
            ORDER BY total $order
            LIMIT ?";
    if ($routerFilter) {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("sssi", $start, $end, $routerParam, $limit);
    } else {
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ssi", $start, $end, $limit);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $profiles = [];
    while ($row = $result->fetch_assoc()) {
        $profiles[] = $row;
    }
    $stmt->close();
    return $profiles;
}

// Top 3 and Bottom 3 for this month
$topProfilesThisMonth = getProfileSales($conn, $thisMonthStart, $thisMonthEnd, 'DESC', 3, $routerFilter, $routerParam);
$bottomProfilesThisMonth = getProfileSales($conn, $thisMonthStart, $thisMonthEnd, 'ASC', 3, $routerFilter, $routerParam);

// Top 3 and Bottom 3 for last month
$topProfilesLastMonth = getProfileSales($conn, $lastMonthStart, $lastMonthEnd, 'DESC', 3, $routerFilter, $routerParam);
$bottomProfilesLastMonth = getProfileSales($conn, $lastMonthStart, $lastMonthEnd, 'ASC', 3, $routerFilter, $routerParam);

// Helper to get monthly revenue for a profile for the last 12 months
function getProfileMonthlyRevenue($conn, $profile, $routerFilter = '', $routerParam = '') {
    $months = [];
    for ($i = 11; $i >= 0; $i--) {
        $start = date('Y-m-01', strtotime("-$i months"));
        $end = date('Y-m-01', strtotime("-" . ($i - 1) . " months"));
        $sql = "SELECT SUM(amount) as total FROM transaction_history
                WHERE status='Paid' AND profile=? AND created_at >= ? AND created_at < ? $routerFilter";
        $stmt = $conn->prepare($sql);
        if ($routerFilter) {
            $stmt->bind_param("ssss", $profile, $start, $end, $routerParam);
        } else {
            $stmt->bind_param("sss", $profile, $start, $end);
        }
        $stmt->execute();
        $stmt->bind_result($total);
        $stmt->fetch();
        $months[] = floatval($total);
        $stmt->close();
    }
    return $months;
}

// Helper to get total revenue for a profile for a given month
function getProfileRevenue($conn, $profile, $start, $end, $routerFilter = '', $routerParam = '') {
    $sql = "SELECT SUM(amount) as total FROM transaction_history
            WHERE status='Paid' AND profile=? AND created_at >= ? AND created_at < ? $routerFilter";
    $stmt = $conn->prepare($sql);
    if ($routerFilter) {
        $stmt->bind_param("ssss", $profile, $start, $end, $routerParam);
    } else {
        $stmt->bind_param("sss", $profile, $start, $end);
    }
    $stmt->execute();
    $stmt->bind_result($total);
    $stmt->fetch();
    $stmt->close();
    return floatval($total);
}

// Prepare data for each profile (top and bottom, this month)
$profileCards = [];
foreach (array_merge($topProfilesThisMonth, $bottomProfilesThisMonth) as $profileRow) {
    $profile = $profileRow['profile'];
    $monthlyRevenue = getProfileMonthlyRevenue($conn, $profile, $routerFilter, $routerParam);
    $thisMonthRevenue = end($monthlyRevenue);
    $lastMonthRevenue = $monthlyRevenue[count($monthlyRevenue) - 2];
    $growth = percentGrowth($thisMonthRevenue, $lastMonthRevenue);
    $profileCards[] = [
        'profile' => $profile,
        'this_month_revenue' => $thisMonthRevenue,
        'growth' => $growth,
        'monthly_revenue' => $monthlyRevenue,
    ];
}
?>


<!DOCTYPE html>
<html lang="en" dir="ltr">
    <head>
        <meta charset="utf-8">
        <meta http-equiv="X-UA-Compatible" content="IE=edge">
        <title>Ewusiwa Hostel Wifi</title>
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <link rel="icon" type="image/x-icon" href="favicon.png">
        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin="">
        <link href="css2?family=Nunito:wght@400;500;600;700;800&display=swap" rel="stylesheet">
        <link rel="stylesheet" type="text/css" media="screen" href="assets/css/perfect-scrollbar.min.css">
        <link rel="stylesheet" type="text/css" media="screen" href="assets/css/style.css">
        <link defer="" rel="stylesheet" type="text/css" media="screen" href="assets/css/animate.css">
        <script src="assets/js/perfect-scrollbar.min.js"></script>
        <script defer="" src="assets/js/popper.min.js"></script>
        <script defer="" src="assets/js/tippy-bundle.umd.min.js"></script>
        <script defer="" src="assets/js/sweetalert.min.js"></script>
    </head>

    <body x-data="main" class="relative overflow-x-hidden font-nunito text-sm font-normal antialiased" :class="[ $store.app.sidebar ? 'toggle-sidebar' : '', $store.app.theme === 'dark' || $store.app.isDarkMode ?  'dark' : '', $store.app.menu, $store.app.layout,$store.app.rtlClass]">
        <!-- sidebar menu overlay -->
        <div x-cloak="" class="fixed inset-0 z-50 bg-[black]/60 lg:hidden" :class="{'hidden' : !$store.app.sidebar}" @click="$store.app.toggleSidebar()"></div>

        <!-- screen loader -->
        <div class="screen_loader animate__animated fixed inset-0 z-[60] grid place-content-center bg-[#fafafa] dark:bg-[#060818]">
            <img src="logo.jpg" alt="Loading..." width="96" height="96" class="animate-pulse rounded-full shadow-lg" />
        </div>

        <!-- scroll to top button -->
        <div class="fixed bottom-6 z-50 ltr:right-6 rtl:left-6" x-data="scrollToTop">
            <template x-if="showTopButton">
                <button type="button" class="btn btn-outline-primary animate-pulse rounded-full bg-[#fafafa] p-2 dark:bg-[#060818] dark:hover:bg-primary" @click="goToTop">
                    <svg width="24" height="24" class="h-4 w-4" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path opacity="0.5" fill-rule="evenodd" clip-rule="evenodd" d="M12 20.75C12.4142 20.75 12.75 20.4142 12.75 20L12.75 10.75L11.25 10.75L11.25 20C11.25 20.4142 11.5858 20.75 12 20.75Z" fill="currentColor"></path>
                        <path d="M6.00002 10.75C5.69667 10.75 5.4232 10.5673 5.30711 10.287C5.19103 10.0068 5.25519 9.68417 5.46969 9.46967L11.4697 3.46967C11.6103 3.32902 11.8011 3.25 12 3.25C12.1989 3.25 12.3897 3.32902 12.5304 3.46967L18.5304 9.46967C18.7449 9.68417 18.809 10.0068 18.6929 10.287C18.5768 10.5673 18.3034 10.75 18 10.75L6.00002 10.75Z" fill="currentColor"></path>
                    </svg>
                </button>
            </template>
        </div>

        <div class="main-container min-h-screen text-black dark:text-white-dark" :class="[$store.app.navbar]">
            <!-- start sidebar section -->
        <?php include 'sidebar.php'; ?>
            <!-- end sidebar section -->

            <div class="main-content flex flex-col min-h-screen">
                <!-- start header section -->
                <header class="z-40" :class="{'dark' : $store.app.semidark && $store.app.menu === 'horizontal'}">
                    <div class="shadow-sm">
                        <div class="relative flex w-full items-center bg-white px-5 py-2.5 dark:bg-[#0e1726]">
                            <div class="horizontal-logo flex items-center justify-between ltr:mr-2 rtl:ml-2 lg:hidden">
                                <a href="index.php" class="main-logo flex shrink-0 items-center">
                                    <img class="inline w-8 ltr:-ml-1 rtl:-mr-1" src="assets/images/logo.png" alt="image">
                                    <span class="hidden align-middle text-2xl font-semibold transition-all duration-300 ltr:ml-1.5 rtl:mr-1.5 dark:text-white-light md:inline">Ewusiwa Hostel Wifi</span>
                                </a>

                                <a href="javascript:;" class="collapse-icon flex flex-none rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary ltr:ml-2 rtl:mr-2 dark:bg-dark/40 dark:text-[#d0d2d6] dark:hover:bg-dark/60 dark:hover:text-primary lg:hidden" @click="$store.app.toggleSidebar()">
                                    <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M20 7L4 7" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        <path opacity="0.5" d="M20 12L4 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        <path d="M20 17L4 17" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                </a>
                            </div>
                            <div class="hidden ltr:mr-2 rtl:ml-2 sm:block">
                                <ul class="flex items-center space-x-2 rtl:space-x-reverse dark:text-[#d0d2d6]">
                                    <li>
                                        <a href="apps-todolist.php" class="block rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60">
                                            <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path opacity="0.5" d="M22 10.5V12C22 16.714 22 19.0711 20.5355 20.5355C19.0711 22 16.714 22 12 22C7.28595 22 4.92893 22 3.46447 20.5355C2 19.0711 2 16.714 2 12C2 7.28595 2 4.92893 3.46447 3.46447C4.92893 2 7.28595 2 12 2H13.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                <path d="M17.3009 2.80624L16.652 3.45506L10.6872 9.41993C10.2832 9.82394 10.0812 10.0259 9.90743 10.2487C9.70249 10.5114 9.52679 10.7957 9.38344 11.0965C9.26191 11.3515 9.17157 11.6225 8.99089 12.1646L8.41242 13.9L8.03811 15.0229C7.9492 15.2897 8.01862 15.5837 8.21744 15.7826C8.41626 15.9814 8.71035 16.0508 8.97709 15.9619L10.1 15.5876L11.8354 15.0091C12.3775 14.8284 12.6485 14.7381 12.9035 14.6166C13.2043 14.4732 13.4886 14.2975 13.7513 14.0926C13.9741 13.9188 14.1761 13.7168 14.5801 13.3128L20.5449 7.34795L21.1938 6.69914C22.2687 5.62415 22.2687 3.88124 21.1938 2.80624C20.1188 1.73125 18.3759 1.73125 17.3009 2.80624Z" stroke="currentColor" stroke-width="1.5"></path>
                                                <path opacity="0.5" d="M16.6522 3.45508C16.6522 3.45508 16.7333 4.83381 17.9499 6.05034C19.1664 7.26687 20.5451 7.34797 20.5451 7.34797M10.1002 15.5876L8.4126 13.9" stroke="currentColor" stroke-width="1.5"></path>
                                            </svg>
                                        </a>
                                    </li>
                                </ul>
                            </div>
                            <div x-data="header" class="flex items-center space-x-1.5 ltr:ml-auto rtl:mr-auto rtl:space-x-reverse dark:text-[#d0d2d6] sm:flex-1 ltr:sm:ml-0 sm:rtl:mr-0 lg:space-x-2">
                                <div class="sm:ltr:mr-auto sm:rtl:ml-auto" x-data="{ search: false }" @click.outside="search = false">
                                    <form class="absolute inset-x-0 top-1/2 z-10 mx-4 hidden -translate-y-1/2 sm:relative sm:top-0 sm:mx-0 sm:block sm:translate-y-0" :class="{'!block' : search}" @submit.prevent="search = false">
                                        <div class="relative">
                                            <input type="text" class="peer form-input bg-gray-100 placeholder:tracking-widest ltr:pl-9 ltr:pr-9 rtl:pr-9 rtl:pl-9 sm:bg-transparent ltr:sm:pr-4 rtl:sm:pl-4" placeholder="Search...">
                                            <button type="button" class="absolute inset-0 h-9 w-9 appearance-none peer-focus:text-primary ltr:right-auto rtl:left-auto">
                                                <svg class="mx-auto" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="11.5" cy="11.5" r="9.5" stroke="currentColor" stroke-width="1.5" opacity="0.5"></circle>
                                                    <path d="M18.5 18.5L22 22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                </svg>
                                            </button>
                                            <button type="button" class="absolute top-1/2 block -translate-y-1/2 hover:opacity-80 ltr:right-2 rtl:left-2 sm:hidden" @click="search = false">
                                                <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle opacity="0.5" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path d="M14.5 9.50002L9.5 14.5M9.49998 9.5L14.5 14.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                </svg>
                                            </button>
                                        </div>
                                    </form>
                                    <button type="button" class="search_btn rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 dark:bg-dark/40 dark:hover:bg-dark/60 sm:hidden" @click="search = ! search">
                                        <svg class="mx-auto h-4.5 w-4.5 dark:text-[#d0d2d6]" width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="11.5" cy="11.5" r="9.5" stroke="currentColor" stroke-width="1.5" opacity="0.5"></circle>
                                            <path d="M18.5 18.5L22 22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </button>
                                </div>
                                <div>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'light'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('dark')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <circle cx="12" cy="12" r="5" stroke="currentColor" stroke-width="1.5"></circle>
                                            <path d="M12 2V4" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M12 20V22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M4 12L2 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path d="M22 12L20 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M19.7778 4.22266L17.5558 6.25424" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M4.22217 4.22266L6.44418 6.25424" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M6.44434 17.5557L4.22211 19.7779" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M19.7778 19.7773L17.5558 17.5551" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </a>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'dark'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('system')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M21.0672 11.8568L20.4253 11.469L21.0672 11.8568ZM12.1432 2.93276L11.7553 2.29085V2.29085L12.1432 2.93276ZM21.25 12C21.25 17.1086 17.1086 21.25 12 21.25V22.75C17.9371 22.75 22.75 17.9371 22.75 12H21.25ZM12 21.25C6.89137 21.25 2.75 17.1086 2.75 12H1.25C1.25 17.9371 6.06294 22.75 12 22.75V21.25ZM2.75 12C2.75 6.89137 6.89137 2.75 12 2.75V1.25C6.06294 1.25 1.25 6.06294 1.25 12H2.75ZM15.5 14.25C12.3244 14.25 9.75 11.6756 9.75 8.5H8.25C8.25 12.5041 11.4959 15.75 15.5 15.75V14.25ZM20.4253 11.469C19.4172 13.1373 17.5882 14.25 15.5 14.25V15.75C18.1349 15.75 20.4407 14.3439 21.7092 12.2447L20.4253 11.469ZM9.75 8.5C9.75 6.41182 10.8627 4.5828 12.531 3.57467L11.7553 2.29085C9.65609 3.5593 8.25 5.86509 8.25 8.5H9.75ZM12 2.75C11.9115 2.75 11.8077 2.71008 11.7324 2.63168C11.6686 2.56527 11.6538 2.50244 11.6503 2.47703C11.6461 2.44587 11.6482 2.35557 11.7553 2.29085L12.531 3.57467C13.0342 3.27065 13.196 2.71398 13.1368 2.27627C13.0754 1.82126 12.7166 1.25 12 1.25V2.75ZM21.7092 12.2447C21.6444 12.3518 21.5541 12.3539 21.523 12.3497C21.4976 12.3462 21.4347 12.3314 21.3683 12.2676C21.2899 12.1923 21.25 12.0885 21.25 12H22.75C22.75 11.2834 22.1787 10.9246 21.7237 10.8632C21.286 10.804 20.7293 10.9658 20.4253 11.469L21.7092 12.2447Z" fill="currentColor"></path>
                                        </svg>
                                    </a>
                                    <a href="javascript:;" x-cloak="" x-show="$store.app.theme === 'system'" href="javascript:;" class="flex items-center rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="$store.app.toggleTheme('light')">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M3 9C3 6.17157 3 4.75736 3.87868 3.87868C4.75736 3 6.17157 3 9 3H15C17.8284 3 19.2426 3 20.1213 3.87868C21 4.75736 21 6.17157 21 9V14C21 15.8856 21 16.8284 20.4142 17.4142C19.8284 18 18.8856 18 17 18H7C5.11438 18 4.17157 18 3.58579 17.4142C3 16.8284 3 15.8856 3 14V9Z" stroke="currentColor" stroke-width="1.5"></path>
                                            <path opacity="0.5" d="M22 21H2" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                            <path opacity="0.5" d="M15 15H9" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                        </svg>
                                    </a>
                                </div>

                                <div class="dropdown shrink-0" x-data="dropdown" @click.outside="open = false">
                                    <a href="javascript:;" class="block rounded-full bg-white-light/40 p-2 hover:bg-white-light/90 hover:text-primary dark:bg-dark/40 dark:hover:bg-dark/60" @click="toggle">
                                        <img :src="`assets/images/flags/${$store.app.locale.toUpperCase()}.svg`" alt="image" class="h-5 w-5 rounded-full object-cover">
                                    </a>
                                    <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="top-11 grid w-[280px] grid-cols-2 gap-y-2 !px-2 font-semibold text-dark ltr:-right-14 rtl:-left-14 dark:text-white-dark dark:text-white-light/90 sm:ltr:-right-2 sm:rtl:-left-2">
                                        <template x-for="item in languages">
                                            <li>
                                                <a href="javascript:;" class="hover:text-primary" @click="$store.app.toggleLocale(item.value),toggle()" :class="{'bg-primary/10 text-primary' : $store.app.locale == item.value}">
                                                    <img class="h-5 w-5 rounded-full object-cover" :src="`assets/images/flags/${item.value.toUpperCase()}.svg`" alt="image">
                                                    <span class="ltr:ml-3 rtl:mr-3" x-text="item.key"></span>
                                                </a>
                                            </li>
                                        </template>
                                    </ul>
                                </div>

                                <div class="dropdown flex-shrink-0" x-data="dropdown" @click.outside="open = false">
                                    <a href="javascript:;" class="group relative" @click="toggle()">
                                        <span><img class="h-9 w-9 rounded-full object-cover saturate-50 group-hover:saturate-100" src="assets/images/user-profile.jpeg" alt="image"></span>
                                    </a>
                                    <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="top-11 w-[230px] !py-0 font-semibold text-dark ltr:right-0 rtl:left-0 dark:text-white-dark dark:text-white-light/90">
                                        <li>
                                            <div class="flex items-center px-4 py-4">
                                                <div class="flex-none">
                                                    <img class="h-10 w-10 rounded-md object-cover" src="assets/images/user-profile.jpeg" alt="image">
                                                </div>
                                                <div class="truncate ltr:pl-4 rtl:pr-4">
                                                    <h4 class="text-base">
                                                        Ewusiwa Hostel Wifi
                                                </div>
                                            </div>
                                        </li>
                                        <li>
                                            <a href="users-profile.php" class="dark:hover:text-white" @click="toggle">
                                                <svg class="h-4.5 w-4.5 shrink-0 ltr:mr-2 rtl:ml-2" width="18" height="18" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <circle cx="12" cy="6" r="4" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path opacity="0.5" d="M20 17.5C20 19.9853 20 22 12 22C4 22 4 19.9853 4 17.5C4 15.0147 7.58172 13 12 13C16.4183 13 20 15.0147 20 17.5Z" stroke="currentColor" stroke-width="1.5"></path>
                                                </svg>
                                                Profile</a>
                                        </li>
                                        <li class="border-t border-white-light dark:border-white-light/10">
                                            <a href="index.php?logout=1" class="!py-3 text-danger" @click="toggle">
                                                <svg class="h-4.5 w-4.5 rotate-90 ltr:mr-2 rtl:ml-2" width="18" height="18" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path opacity="0.5" d="M17 9.00195C19.175 9.01406 20.3529 9.11051 21.1213 9.8789C22 10.7576 22 12.1718 22 15.0002V16.0002C22 18.8286 22 20.2429 21.1213 21.1215C20.2426 22.0002 18.8284 22.0002 16 22.0002H8C5.17157 22.0002 3.75736 22.0002 2.87868 21.1215C2 20.2429 2 18.8286 2 16.0002L2 15.0002C2 12.1718 2 10.7576 2.87868 9.87889C3.64706 9.11051 4.82497 9.01406 7 9.00195" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <path d="M12 15L12 2M12 2L15 5.5M12 2L9 5.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                                Sign Out
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <!-- horizontal menu -->
                        <ul class="horizontal-menu hidden border-t border-[#ebedf2] bg-white py-1.5 px-6 font-semibold text-black rtl:space-x-reverse dark:border-[#191e3a] dark:bg-[#0e1726] dark:text-white-dark lg:space-x-1.5 xl:space-x-8">
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link active">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M2 12.2039C2 9.91549 2 8.77128 2.5192 7.82274C3.0384 6.87421 3.98695 6.28551 5.88403 5.10813L7.88403 3.86687C9.88939 2.62229 10.8921 2 12 2C13.1079 2 14.1106 2.62229 16.116 3.86687L18.116 5.10812C20.0131 6.28551 20.9616 6.87421 21.4808 7.82274C22 8.77128 22 9.91549 22 12.2039V13.725C22 17.6258 22 19.5763 20.8284 20.7881C19.6569 22 17.7712 22 14 22H10C6.22876 22 4.34315 22 3.17157 20.7881C2 19.5763 2 17.6258 2 13.725V12.2039Z" fill="currentColor"></path>
                                            <path d="M9 17.25C8.58579 17.25 8.25 17.5858 8.25 18C8.25 18.4142 8.58579 18.75 9 18.75H15C15.4142 18.75 15.75 18.4142 15.75 18C15.75 17.5858 15.4142 17.25 15 17.25H9Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Dashboard</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="index.php" class="active">Sales</a>
                                    </li>
                                    <li>
                                        <a href="finance.php">Finance</a>
                                    </li>
                                    
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <g opacity="0.5">
                                                <path d="M14 2.75C15.9068 2.75 17.2615 2.75159 18.2892 2.88976C19.2952 3.02503 19.8749 3.27869 20.2981 3.7019C20.7213 4.12511 20.975 4.70476 21.1102 5.71085C21.2484 6.73851 21.25 8.09318 21.25 10C21.25 10.4142 21.5858 10.75 22 10.75C22.4142 10.75 22.75 10.4142 22.75 10V9.94359C22.75 8.10583 22.75 6.65019 22.5969 5.51098C22.4392 4.33856 22.1071 3.38961 21.3588 2.64124C20.6104 1.89288 19.6614 1.56076 18.489 1.40314C17.3498 1.24997 15.8942 1.24998 14.0564 1.25H14C13.5858 1.25 13.25 1.58579 13.25 2C13.25 2.41421 13.5858 2.75 14 2.75Z" fill="currentColor"></path>
                                                <path d="M9.94358 1.25H10C10.4142 1.25 10.75 1.58579 10.75 2C10.75 2.41421 10.4142 2.75 10 2.75C8.09318 2.75 6.73851 2.75159 5.71085 2.88976C4.70476 3.02503 4.12511 3.27869 3.7019 3.7019C3.27869 4.12511 3.02503 4.70476 2.88976 5.71085C2.75159 6.73851 2.75 8.09318 2.75 10C2.75 10.4142 2.41421 10.75 2 10.75C1.58579 10.75 1.25 10.4142 1.25 10V9.94358C1.24998 8.10583 1.24997 6.65019 1.40314 5.51098C1.56076 4.33856 1.89288 3.38961 2.64124 2.64124C3.38961 1.89288 4.33856 1.56076 5.51098 1.40314C6.65019 1.24997 8.10583 1.24998 9.94358 1.25Z" fill="currentColor"></path>
                                                <path d="M22 13.25C22.4142 13.25 22.75 13.5858 22.75 14V14.0564C22.75 15.8942 22.75 17.3498 22.5969 18.489C22.4392 19.6614 22.1071 20.6104 21.3588 21.3588C20.6104 22.1071 19.6614 22.4392 18.489 22.5969C17.3498 22.75 15.8942 22.75 14.0564 22.75H14C13.5858 22.75 13.25 22.4142 13.25 22C13.25 21.5858 13.5858 21.25 14 21.25C15.9068 21.25 17.2615 21.2484 18.2892 21.1102C19.2952 20.975 19.8749 20.7213 20.2981 20.2981C20.7213 19.8749 20.975 19.2952 21.1102 18.2892C21.2484 17.2615 21.25 15.9068 21.25 14C21.25 13.5858 21.5858 13.25 22 13.25Z" fill="currentColor"></path>
                                                <path d="M2.75 14C2.75 13.5858 2.41421 13.25 2 13.25C1.58579 13.25 1.25 13.5858 1.25 14V14.0564C1.24998 15.8942 1.24997 17.3498 1.40314 18.489C1.56076 19.6614 1.89288 20.6104 2.64124 21.3588C3.38961 22.1071 4.33856 22.4392 5.51098 22.5969C6.65019 22.75 8.10583 22.75 9.94359 22.75H10C10.4142 22.75 10.75 22.4142 10.75 22C10.75 21.5858 10.4142 21.25 10 21.25C8.09318 21.25 6.73851 21.2484 5.71085 21.1102C4.70476 20.975 4.12511 20.7213 3.7019 20.2981C3.27869 19.8749 3.02503 19.2952 2.88976 18.2892C2.75159 17.2615 2.75 15.9068 2.75 14Z" fill="currentColor"></path>
                                            </g>
                                            <path d="M5.52721 5.52721C5 6.05442 5 6.90294 5 8.6C5 9.73137 5 10.2971 5.35147 10.6485C5.70294 11 6.26863 11 7.4 11H8.6C9.73137 11 10.2971 11 10.6485 10.6485C11 10.2971 11 9.73137 11 8.6V7.4C11 6.26863 11 5.70294 10.6485 5.35147C10.2971 5 9.73137 5 8.6 5C6.90294 5 6.05442 5 5.52721 5.52721Z" fill="currentColor"></path>
                                            <path d="M5.52721 18.4728C5 17.9456 5 17.0971 5 15.4C5 14.2686 5 13.7029 5.35147 13.3515C5.70294 13 6.26863 13 7.4 13H8.6C9.73137 13 10.2971 13 10.6485 13.3515C11 13.7029 11 14.2686 11 15.4V16.6C11 17.7314 11 18.2971 10.6485 18.6485C10.2971 19 9.73138 19 8.60002 19C6.90298 19 6.05441 19 5.52721 18.4728Z" fill="currentColor"></path>
                                            <path d="M13 7.4C13 6.26863 13 5.70294 13.3515 5.35147C13.7029 5 14.2686 5 15.4 5C17.0971 5 17.9456 5 18.4728 5.52721C19 6.05442 19 6.90294 19 8.6C19 9.73137 19 10.2971 18.6485 10.6485C18.2971 11 17.7314 11 16.6 11H15.4C14.2686 11 13.7029 11 13.3515 10.6485C13 10.2971 13 9.73137 13 8.6V7.4Z" fill="currentColor"></path>
                                            <path d="M13.3515 18.6485C13 18.2971 13 17.7314 13 16.6V15.4C13 14.2686 13 13.7029 13.3515 13.3515C13.7029 13 14.2686 13 15.4 13H16.6C17.7314 13 18.2971 13 18.6485 13.3515C19 13.7029 19 14.2686 19 15.4C19 17.097 19 17.9456 18.4728 18.4728C17.9456 19 17.0971 19 15.4 19C14.2687 19 13.7029 19 13.3515 18.6485Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Apps</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="apps-todolist.php">Todo List</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path d="M4.97883 9.68508C2.99294 8.89073 2 8.49355 2 8C2 7.50645 2.99294 7.10927 4.97883 6.31492L7.7873 5.19153C9.77318 4.39718 10.7661 4 12 4C13.2339 4 14.2268 4.39718 16.2127 5.19153L19.0212 6.31492C21.0071 7.10927 22 7.50645 22 8C22 8.49355 21.0071 8.89073 19.0212 9.68508L16.2127 10.8085C14.2268 11.6028 13.2339 12 12 12C10.7661 12 9.77318 11.6028 7.7873 10.8085L4.97883 9.68508Z" fill="currentColor"></path>
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M2 8C2 8.49355 2.99294 8.89073 4.97883 9.68508L7.7873 10.8085C9.77318 11.6028 10.7661 12 12 12C13.2339 12 14.2268 11.6028 16.2127 10.8085L19.0212 9.68508C21.0071 8.89073 22 8.49355 22 8C22 7.50645 21.0071 7.10927 19.0212 6.31492L16.2127 5.19153C14.2268 4.39718 13.2339 4 12 4C10.7661 4 9.77318 4.39718 7.7873 5.19153L4.97883 6.31492C2.99294 7.10927 2 7.50645 2 8Z" fill="currentColor"></path>
                                            <path opacity="0.7" d="M5.76613 10L4.97883 10.3149C2.99294 11.1093 2 11.5065 2 12C2 12.4935 2.99294 12.8907 4.97883 13.6851L7.7873 14.8085C9.77318 15.6028 10.7661 16 12 16C13.2339 16 14.2268 15.6028 16.2127 14.8085L19.0212 13.6851C21.0071 12.8907 22 12.4935 22 12C22 11.5065 21.0071 11.1093 19.0212 10.3149L18.2339 10L16.2127 10.8085C14.2268 11.6028 13.2339 12 12 12C10.7661 12 9.77318 11.6028 7.7873 10.8085L5.76613 10Z" fill="currentColor"></path>
                                            <path opacity="0.4" d="M5.76613 14L4.97883 14.3149C2.99294 15.1093 2 15.5065 2 16C2 16.4935 2.99294 16.8907 4.97883 17.6851L7.7873 18.8085C9.77318 19.6028 10.7661 20 12 20C13.2339 20 14.2268 19.6028 16.2127 18.8085L19.0212 17.6851C21.0071 16.8907 22 16.4935 22 16C22 15.5065 21.0071 15.1093 19.0212 14.3149L18.2339 14L16.2127 14.8085C14.2268 15.6028 13.2339 16 12 16C10.7661 16 9.77318 15.6028 7.7873 14.8085L5.76613 14Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Tables</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li class="relative">
                                        <a href="javascript:;">Data Tables
                                            <div class="ltr:ml-auto rtl:mr-auto rtl:rotate-180">
                                                <svg width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                        </a>
                                        <ul class="absolute top-0 z-[10] hidden min-w-[180px] rounded bg-white p-0 py-2 text-dark shadow ltr:left-[95%] rtl:right-[95%] dark:bg-[#1b2e4b] dark:text-white-dark">
                                            <li>
                                                <a href="datatables-multiple-tables.php">Multiple Tables</a>
                                            </li>
                                        </ul>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M3 10C3 6.22876 3 4.34315 4.17157 3.17157C5.34315 2 7.22876 2 11 2H13C16.7712 2 18.6569 2 19.8284 3.17157C21 4.34315 21 6.22876 21 10V14C21 17.7712 21 19.6569 19.8284 20.8284C18.6569 22 16.7712 22 13 22H11C7.22876 22 5.34315 22 4.17157 20.8284C3 19.6569 3 17.7712 3 14V10Z" fill="currentColor"></path>
                                            <path d="M16.5189 16.5013C16.6939 16.3648 16.8526 16.2061 17.1701 15.8886L21.1275 11.9312C21.2231 11.8356 21.1793 11.6708 21.0515 11.6264C20.5844 11.4644 19.9767 11.1601 19.4083 10.5917C18.8399 10.0233 18.5356 9.41561 18.3736 8.94849C18.3292 8.82066 18.1644 8.77687 18.0688 8.87254L14.1114 12.8299C13.7939 13.1474 13.6352 13.3061 13.4987 13.4811C13.3377 13.6876 13.1996 13.9109 13.087 14.1473C12.9915 14.3476 12.9205 14.5606 12.7786 14.9865L12.5951 15.5368L12.3034 16.4118L12.0299 17.2323C11.9601 17.4419 12.0146 17.6729 12.1708 17.8292C12.3271 17.9854 12.5581 18.0399 12.7677 17.9701L13.5882 17.6966L14.4632 17.4049L15.0135 17.2214L15.0136 17.2214C15.4394 17.0795 15.6524 17.0085 15.8527 16.913C16.0891 16.8004 16.3124 16.6623 16.5189 16.5013Z" fill="currentColor"></path>
                                            <path d="M22.3665 10.6922C23.2112 9.84754 23.2112 8.47812 22.3665 7.63348C21.5219 6.78884 20.1525 6.78884 19.3078 7.63348L19.1806 7.76071C19.0578 7.88348 19.0022 8.05496 19.0329 8.22586C19.0522 8.33336 19.0879 8.49053 19.153 8.67807C19.2831 9.05314 19.5288 9.54549 19.9917 10.0083C20.4545 10.4712 20.9469 10.7169 21.3219 10.847C21.5095 10.9121 21.6666 10.9478 21.7741 10.9671C21.945 10.9978 22.1165 10.9422 22.2393 10.8194L22.3665 10.6922Z" fill="currentColor"></path>
                                            <path fill-rule="evenodd" clip-rule="evenodd" d="M7.25 9C7.25 8.58579 7.58579 8.25 8 8.25H14.5C14.9142 8.25 15.25 8.58579 15.25 9C15.25 9.41421 14.9142 9.75 14.5 9.75H8C7.58579 9.75 7.25 9.41421 7.25 9ZM7.25 13C7.25 12.5858 7.58579 12.25 8 12.25H11C11.4142 12.25 11.75 12.5858 11.75 13C11.75 13.4142 11.4142 13.75 11 13.75H8C7.58579 13.75 7.25 13.4142 7.25 13ZM7.25 17C7.25 16.5858 7.58579 16.25 8 16.25H9.5C9.91421 16.25 10.25 16.5858 10.25 17C10.25 17.4142 9.91421 17.75 9.5 17.75H8C7.58579 17.75 7.25 17.4142 7.25 17Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">Settings</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <ul class="sub-menu">
                                    <li>
                                        <a href="forms-layouts.php">Layouts</a>
                                    </li>
                                </ul>
                            </li>
                            <li class="menu nav-item relative">
                                <a href="javascript:;" class="nav-link">
                                    <div class="flex items-center">
                                        <svg width="20" height="20" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="shrink-0">
                                            <path opacity="0.5" d="M22 12C22 17.5228 17.5228 22 12 22C6.47715 22 2 17.5228 2 12C2 6.47715 6.47715 2 12 2C17.5228 2 22 6.47715 22 12Z" fill="currentColor"></path>
                                            <path d="M12.75 9C12.75 8.58579 12.4142 8.25 12 8.25C11.5858 8.25 11.25 8.58579 11.25 9L11.25 11.25H9C8.58579 11.25 8.25 11.5858 8.25 12C8.25 12.4142 8.58579 12.75 9 12.75H11.25V15C11.25 15.4142 11.5858 15.75 12 15.75C12.4142 15.75 12.75 15.4142 12.75 15L12.75 12.75H15C15.4142 12.75 15.75 12.4142 15.75 12C15.75 11.5858 15.4142 11.25 15 11.25H12.75V9Z" fill="currentColor"></path>
                                        </svg>
                                        <span class="px-1">More</span>
                                    </div>
                                    <div class="right_arrow">
                                        <svg class="h-4 w-4 rotate-90" width="16" height="16" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M9 5L15 12L9 19" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                        </svg>
                                    </div>
                                </a>
                                <!-- <ul class="sub-menu">
                                    <li>
                                        <a href="charts.php">Charts</a>
                                    </li>
                                </ul> -->
                            </li>
                        </ul>
                    </div>
                </header>
                <!-- end header section -->

                <div class="animate__animated p-6" :class="[$store.app.animation]">
                    <div x-data="finance">
                        <!-- ...existing HTML code... -->
                        <!-- Add this where you want the unit selector to appear -->
                        <ul class="flex space-x-2 rtl:space-x-reverse">
                            <li>
                                <a href="?unit=All Units" class="<?= $selectedUnit === 'All Units' ? 'text-primary underline' : '' ?>">All Units</a>
                            </li>
                            <!--<li class="before:content-['/'] ltr:before:mr-1 rtl:before:ml-1">-->
                            <!--    <a href="?unit=CCTU" class="<?= $selectedUnit === 'CCTU' ? 'text-primary underline' : '' ?>">CCTU</a>-->
                            <!--</li>-->
                            <!--<li class="before:content-['/'] ltr:before:mr-1 rtl:before:ml-1">-->
                            <!--    <a href="?unit=Accra" class="<?= $selectedUnit === 'Accra' ? 'text-primary underline' : '' ?>">ACCRA</a>-->
                            <!--</li>-->
                        </ul>
                        <!-- ...existing HTML code... -->
                        <div class="pt-5">
                            <div class="mb-6 grid grid-cols-1 gap-6 text-white sm:grid-cols-2 xl:grid-cols-4">
                                <!-- Revenue Today -->
                                    <div class="panel bg-gradient-to-r from-cyan-500 to-cyan-400">
                                        <div class="flex justify-between">
                                            <div class="text-md font-semibold ltr:mr-1 rtl:ml-1">Revenue Today</div>
                                            <!-- ...dropdown... -->
                                        </div>
                                        <div class="mt-5 flex items-center">
                                            <div class="text-3xl font-bold ltr:mr-3 rtl:ml-3">GH¢ <?php echo number_format($revenue_today, 2); ?></div>
                                            <div class="badge bg-white/30"><?php echo ($growth_today >= 0 ? '+ ' : '- ') . abs($growth_today) . '%'; ?></div>
                                        </div>
                                        <div class="mt-5 flex items-center font-semibold">
                                            <svg width="24" height="24" ...></svg>
                                            Yesterday GH¢ <?php echo number_format($revenue_yesterday, 2); ?>
                                        </div>
                                    </div>

                                    <!-- Revenue This Week -->
                                    <div class="panel bg-gradient-to-r from-violet-500 to-violet-400">
                                        <div class="flex justify-between">
                                            <div class="text-md font-semibold ltr:mr-1 rtl:ml-1">Revenue This Week</div>
                                            <!-- ...dropdown... -->
                                        </div>
                                        <div class="mt-5 flex items-center">
                                            <div class="text-3xl font-bold ltr:mr-3 rtl:ml-3">GH¢ <?php echo number_format($revenue_this_week, 2); ?></div>
                                            <div class="badge bg-white/30"><?php echo ($growth_week >= 0 ? '+ ' : '- ') . abs($growth_week) . '%'; ?></div>
                                        </div>
                                        <div class="mt-5 flex items-center font-semibold">
                                            <svg width="24" height="24" ...></svg>
                                            Revenue Last Week GH¢ <?php echo number_format($revenue_last_week, 2); ?>
                                        </div>
                                    </div>

                                    <!-- Revenue This Month -->
                                    <div class="panel bg-gradient-to-r from-blue-500 to-blue-400">
                                        <div class="flex justify-between">
                                            <div class="text-md font-semibold ltr:mr-1 rtl:ml-1">Revenue This Month</div>
                                            <!-- ...dropdown... -->
                                        </div>
                                        <div class="mt-5 flex items-center">
                                            <div class="text-3xl font-bold ltr:mr-3 rtl:ml-3">GH¢ <?php echo number_format($revenue_this_month, 2); ?></div>
                                            <div class="badge bg-white/30"><?php echo ($growth_month >= 0 ? '+ ' : '- ') . abs($growth_month) . '%'; ?></div>
                                        </div>
                                        <div class="mt-5 flex items-center font-semibold">
                                            <svg width="24" height="24" ...></svg>
                                            Revenue Last Month GH¢ <?php echo number_format($revenue_last_month, 2); ?>
                                        </div>
                                    </div>

                                    <!-- Revenue This Year -->
                                    <div class="panel bg-gradient-to-r from-fuchsia-500 to-fuchsia-400">
                                        <div class="flex justify-between">
                                            <div class="text-md font-semibold ltr:mr-1 rtl:ml-1">Revenue This Year</div>
                                            <!-- ...dropdown... -->
                                        </div>
                                        <div class="mt-5 flex items-center">
                                            <div class="text-3xl font-bold ltr:mr-3 rtl:ml-3">GH¢ <?php echo number_format($revenue_this_year, 2); ?></div>
                                            <div class="badge bg-white/30"><?php echo ($growth_year >= 0 ? '+ ' : '- ') . abs($growth_year) . '%'; ?></div>
                                        </div>
                                        <div class="mt-5 flex items-center font-semibold">
                                            <svg width="24" height="24" ...></svg>
                                            Revenue Last Year GH¢ <?php echo number_format($revenue_last_year, 2); ?>
                                        </div>
                                    </div>
                            </div>

                            <div class="grid grid-cols-1 gap-6 xl:grid-cols-2">
                                <!-- Top Selling Profiles -->
                                <div>
                                    <div class="mb-5 flex items-center font-bold">
                                        <span class="text-lg">Top Selling Profiles</span>
                                        <a href="javascript:;" class="text-primary hover:text-black ltr:ml-auto rtl:mr-auto dark:hover:text-white-dark">
                                            See All
                                        </a>
                                    </div>
                                    <div class="grid grid-cols-1 gap-6 sm:grid-cols-3 md:mb-5">
                                        <!-- Bitcoin -->
                                        <?php
                                        $topLabels = ['Best Selling', '2nd Selling', '3rd Selling'];
                                        $svgIcons = [
                                            // Bitcoin SVG
                                            '<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve" width="100%" height="100%" version="1.1" shape-rendering="geometricPrecision" text-rendering="geometricPrecision" image-rendering="optimizeQuality" fill-rule="evenodd" clip-rule="evenodd" viewBox="0 0 4091.27 4091.73"><g id="Layer_x0020_1"><g id="_1421344023328"><path fill="#F7931A" fill-rule="nonzero" d="M4030.06 2540.77c-273.24,1096.01 -1383.32,1763.02 -2479.46,1489.71 -1095.68,-273.24 -1762.69,-1383.39 -1489.33,-2479.31 273.12,-1096.13 1383.2,-1763.19 2479,-1489.95 1096.06,273.24 1763.03,1383.51 1489.76,2479.57l0.02 -0.02z"></path><path fill="white" fill-rule="nonzero" d="M2947.77 1754.38c40.72,-272.26 -166.56,-418.61 -450,-516.24l91.95 -368.8 -224.5 -55.94 -89.51 359.09c-59.02,-14.72 -119.63,-28.59 -179.87,-42.34l90.16 -361.46 -224.36 -55.94 -92 368.68c-48.84,-11.12 -96.81,-22.11 -143.35,-33.69l0.26 -1.16 -309.59 -77.31 -59.72 239.78c0,0 166.56,38.18 163.05,40.53 90.91,22.69 107.35,82.87 104.62,130.57l-104.74 420.15c6.26,1.59 14.38,3.89 23.34,7.49 -7.49,-1.86 -15.46,-3.89 -23.73,-5.87l-146.81 588.57c-11.11,27.62 -39.31,69.07 -102.87,53.33 2.25,3.26 -163.17,-40.72 -163.17,-40.72l-111.46 256.98 292.15 72.83c54.35,13.63 107.61,27.89 160.06,41.3l-92.9 373.03 224.24 55.94 92 -369.07c61.26,16.63 120.71,31.97 178.91,46.43l-91.69 367.33 224.51 55.94 92.89 -372.33c382.82,72.45 670.67,43.24 791.83,-303.02 97.63,-278.78 -4.86,-439.58 -206.26,-544.44 146.69,-33.83 257.18,-130.31 286.64,-329.61l-0.07 -0.05zm-512.93 719.26c-69.38,278.78 -538.76,128.08 -690.94,90.29l123.28 -494.2c152.17,37.99 640.17,113.17 567.67,403.91zm69.43 -723.3c-63.29,253.58 -453.96,124.75 -580.69,93.16l111.77 -448.21c126.73,31.59 534.85,90.55 468.94,355.05l-0.02 0z"></path></g></g></svg>',
                                            // Ethereum SVG
                                            '<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve" width="100%" height="100%" version="1.1" shape-rendering="geometricPrecision" text-rendering="geometricPrecision" image-rendering="optimizeQuality" fill-rule="evenodd" clip-rule="evenodd" viewBox="0 0 784.37 1277.39"><g id="Layer_x0020_1"><g id="_1421394342400"><g><polygon fill="#343434" fill-rule="nonzero" points="392.07,0 383.5,29.11 383.5,873.74 392.07,882.29 784.13,650.54 "></polygon><polygon fill="#8C8C8C" fill-rule="nonzero" points="392.07,0 -0,650.54 392.07,882.29 392.07,472.33 "></polygon><polygon fill="#3C3C3B" fill-rule="nonzero" points="392.07,956.52 387.24,962.41 387.24,1263.28 392.07,1277.38 784.37,724.89 "></polygon><polygon fill="#8C8C8C" fill-rule="nonzero" points="392.07,1277.38 392.07,956.52 -0,724.89 "></polygon><polygon fill="#141414" fill-rule="nonzero" points="392.07,882.29 784.13,650.54 392.07,472.33 "></polygon><polygon fill="#393939" fill-rule="nonzero" points="0,650.54 392.07,882.29 392.07,472.33 "></polygon></g></g></g></svg>',
                                            // Litecoin SVG
                                            '<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewBox="0.847 0.876 329.254 329.256"><title>Litecoin</title><path d="M330.102 165.503c0 90.922-73.705 164.629-164.626 164.629C74.554 330.132.848 256.425.848 165.503.848 74.582 74.554.876 165.476.876c90.92 0 164.626 73.706 164.626 164.627" fill="#345d9d"></path><path d="M295.15 165.505c0 71.613-58.057 129.675-129.674 129.675-71.616 0-129.677-58.062-129.677-129.675 0-71.619 58.061-129.677 129.677-129.677 71.618 0 129.674 58.057 129.674 129.677" fill="#345d9d"></path><path d="M155.854 209.482l10.693-40.264 25.316-9.249 6.297-23.663-.215-.587-24.92 9.104 17.955-67.608h-50.921l-23.481 88.23-19.605 7.162-6.478 24.395 19.59-7.156-13.839 51.998h135.521l8.688-32.362h-84.601" fill="#fff"></path></svg>',
                                        ];
                                        foreach ($topProfilesThisMonth as $i => $profile) {
                                            $monthlyRevenue = getProfileMonthlyRevenue($conn, $profile['profile']);
                                            $thisMonthRevenue = end($monthlyRevenue);
                                            $lastMonthRevenue = $monthlyRevenue[count($monthlyRevenue) - 2];
                                            $growth = percentGrowth($thisMonthRevenue, $lastMonthRevenue);
                                            $profileName = htmlspecialchars($profile['profile']);
                                            $growthClass = $growth >= 0 ? 'text-success' : 'text-danger';
                                            $growthSign = $growth >= 0 ? '+' : '';
                                            $xRef = ['bitcoin', 'ethereum', 'litecoin'][$i];
                                        ?>
                                        <div class="panel">
                                            <div class="mb-5 flex items-center font-semibold">
                                                <div class="grid h-10 w-10 shrink-0 place-content-center rounded-full<?= $i == 1 ? ' bg-warning p-2' : '' ?>">
                                                    <?= $svgIcons[$i] ?>
                                                </div>
                                                <div class="ltr:ml-2 rtl:mr-2">
                                                    <h6 class="text-dark dark:text-white-light"><?= $topLabels[$i] ?></h6>
                                                    <p class="text-xs text-white-dark"><?= $profileName ?></p>
                                                </div>
                                            </div>
                                            <div class="mb-5 overflow-hidden">
                                                <div x-ref="<?= $xRef ?>"></div>
                                            </div>
                                            <div class="flex items-center justify-between text-base font-bold">
                                                GH¢ <?= number_format($thisMonthRevenue, 2) ?>
                                                <span class="text-sm font-normal <?= $growthClass ?>"><?= $growthSign . $growth ?>%</span>
                                            </div>
                                        </div>
                                        <?php } ?>
                                    </div>
                                </div>

                                <!-- Least Selling Profiles -->
                                <div>
                                    <div class="mb-5 flex items-center font-bold">
                                        <span class="text-lg">Least Selling Profiles</span>
                                        <a href="javascript:;" class="text-primary hover:text-black ltr:ml-auto rtl:mr-auto dark:hover:text-white-dark">
                                            See All
                                        </a>
                                    </div>
                                    <div class="mb-6 grid grid-cols-1 gap-6 sm:grid-cols-3">
                                        <?php
                                        $leastLabels = ['Least Selling', '2nd Least Selling', '3rd Least Selling'];
                                        $svgIcons = [
                                            // Binance SVG
                                            '<svg width="100%" height="100%" viewbox="0 0 1024 1024" xmlns="http://www.w3.org/2000/svg"><g id="Icon"><circle cx="512" cy="512" r="512" style="fill: #f3ba2f"></circle><path class="st1 fill-white" d="M404.9 468 512 360.9l107.1 107.2 62.3-62.3L512 236.3 342.6 405.7z"></path><path transform="rotate(-45.001 298.629 511.998)" class="st1 fill-white" d="M254.6 467.9h88.1V556h-88.1z"></path><path class="st1 fill-white" d="M404.9 556 512 663.1l107.1-107.2 62.4 62.3h-.1L512 787.7 342.6 618.3l-.1-.1z"></path><path transform="rotate(-45.001 725.364 512.032)" class="st1 fill-white" d="M681.3 468h88.1v88.1h-88.1z"></path><path class="st1 fill-white" d="M575.2 512 512 448.7l-46.7 46.8-5.4 5.3-11.1 11.1-.1.1.1.1 63.2 63.2 63.2-63.3z"></path></g></svg>',
                                            // Tether SVG
                                            '<svg xmlns="http://www.w3.org/2000/svg" width="100%" height="100%" viewbox="0 0 2000 2000"><path d="M1000,0c552.26,0,1000,447.74,1000,1000S1552.24,2000,1000,2000,0,1552.38,0,1000,447.68,0,1000,0" fill="#53ae94"></path><path d="M1123.42,866.76V718H1463.6V491.34H537.28V718H877.5V866.64C601,879.34,393.1,934.1,393.1,999.7s208,120.36,484.4,133.14v476.5h246V1132.8c276-12.74,483.48-67.46,483.48-133s-207.48-120.26-483.48-133m0,225.64v-0.12c-6.94.44-42.6,2.58-122,2.58-63.48,0-108.14-1.8-123.88-2.62v0.2C633.34,1081.66,451,1039.12,451,988.22S633.36,894.84,877.62,884V1050.1c16,1.1,61.76,3.8,124.92,3.8,75.86,0,114-3.16,121-3.8V884c243.8,10.86,425.72,53.44,425.72,104.16s-182,93.32-425.72,104.18" fill="#fff"></path></svg>',
                                            // Solana SVG
                                            '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="100%" height="100%" viewbox="0 0 508.07 398.17"><defs><lineargradient id="linear-gradient" x1="463" y1="205.16" x2="182.39" y2="742.62" gradienttransform="translate(0 -198)" gradientunits="userSpaceOnUse"><stop offset="0" stop-color="#00ffa3"></stop><stop offset="1" stop-color="#dc1fff"></stop></lineargradient><lineargradient id="linear-gradient-2" x1="340.31" y1="141.1" x2="59.71" y2="678.57" xlink:href="#linear-gradient"></lineargradient><lineargradient id="linear-gradient-3" x1="401.26" y1="172.92" x2="120.66" y2="710.39" xlink:href="#linear-gradient"></lineargradient></defs><path class="cls-1 fill-[url(#linear-gradient)]" d="M84.53,358.89A16.63,16.63,0,0,1,96.28,354H501.73a8.3,8.3,0,0,1,5.87,14.18l-80.09,80.09a16.61,16.61,0,0,1-11.75,4.86H10.31A8.31,8.31,0,0,1,4.43,439Z" transform="translate(-1.98 -55)"></path><path class="cls-2 fill-[url(#linear-gradient)]" d="M84.53,59.85A17.08,17.08,0,0,1,96.28,55H501.73a8.3,8.3,0,0,1,5.87,14.18l-80.09,80.09a16.61,16.61,0,0,1-11.75,4.86H10.31A8.31,8.31,0,0,1,4.43,140Z" transform="translate(-1.98 -55)"></path><path class="cls-3 fill-[url(#linear-gradient)]" d="M427.51,208.42a16.61,16.61,0,0,0-11.75-4.86H10.31a8.31,8.31,0,0,0-5.88,14.18l80.1,80.09a16.6,16.6,0,0,0,11.75,4.86H501.73a8.3,8.3,0,0,0,5.87-14.18Z" transform="translate(-1.98 -55)"></path></svg>',
                                        ];
                                        $xRefs = ['binance', 'tether', 'solana'];
                                        $chartColors = ['#e7515a', '#00ab55', '#e7515a'];
                                        foreach ($bottomProfilesThisMonth as $i => $profile) {
                                            $monthlyRevenue = getProfileMonthlyRevenue($conn, $profile['profile']);
                                            $thisMonthRevenue = end($monthlyRevenue);
                                            $lastMonthRevenue = $monthlyRevenue[count($monthlyRevenue) - 2];
                                            $growth = percentGrowth($thisMonthRevenue, $lastMonthRevenue);
                                            $profileName = htmlspecialchars($profile['profile']);
                                            $growthClass = $growth >= 0 ? 'text-success' : 'text-danger';
                                            $growthSign = $growth >= 0 ? '+' : '';
                                        ?>
                                        <div class="panel">
                                            <div class="mb-5 flex items-center font-semibold">
                                                <div class="grid h-10 w-10 shrink-0 place-content-center rounded-full<?= $i == 2 ? ' bg-warning p-2' : '' ?>">
                                                    <?= $svgIcons[$i] ?>
                                                </div>
                                                <div class="ltr:ml-2 rtl:mr-2">
                                                    <h6 class="text-dark dark:text-white-light"><?= $leastLabels[$i] ?></h6>
                                                    <p class="text-xs text-white-dark"><?= $profileName ?></p>
                                                </div>
                                            </div>
                                            <div class="mb-5 overflow-hidden">
                                                <div x-ref="<?= $xRefs[$i] ?>"></div>
                                            </div>
                                            <div class="flex items-center justify-between text-base font-bold">
                                                GH¢ <?= number_format($thisMonthRevenue, 2) ?>
                                                <span class="text-sm font-normal <?= $growthClass ?>"><?= $growthSign . $growth ?>%</span>
                                            </div>
                                        </div>
                                        <?php } ?>
                                    </div>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 gap-6 xl:grid-cols-2">
                                <div class="grid gap-6 xl:grid-flow-row">
                                    <!-- Previous Statement -->
                                    <div class="panel overflow-hidden">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <div class="text-lg font-bold">Previous Statement</div>
                                                <div class="text-success">Paid on June 27, 2022</div>
                                            </div>
                                            <div x-data="dropdown" @click.outside="open = false" class="dropdown">
                                                <a href="javascript:;" @click="toggle">
                                                    <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 opacity-70 hover:opacity-80">
                                                        <circle cx="5" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <circle opacity="0.5" cx="12" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <circle cx="19" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    </svg>
                                                </a>
                                                <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="ltr:right-0 rtl:left-0">
                                                    <li><a href="javascript:;" @click="toggle">View Report</a></li>
                                                    <li><a href="javascript:;" @click="toggle">Edit Report</a></li>
                                                </ul>
                                            </div>
                                        </div>
                                        <div class="relative mt-10">
                                            <div class="absolute -bottom-12 h-24 w-24 ltr:-right-12 rtl:-left-12">
                                                <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="h-full w-full text-success opacity-20">
                                                    <circle opacity="0.5" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path d="M8.5 12.5L10.5 14.5L15.5 9.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                </svg>
                                            </div>
                                            <div class="grid grid-cols-2 gap-6 md:grid-cols-3">
                                                <div>
                                                    <div class="text-primary">Card Limit</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                                <div>
                                                    <div class="text-primary">Spent</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                                <div>
                                                    <div class="text-primary">Minimum</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Current Statement -->
                                    <div class="panel overflow-hidden">
                                        <div class="flex items-center justify-between">
                                            <div>
                                                <div class="text-lg font-bold">Current Statement</div>
                                                <div class="text-danger">Must be paid before July 27, 2022</div>
                                            </div>
                                            <div x-data="dropdown" @click.outside="open = false" class="dropdown">
                                                <a href="javascript:;" @click="toggle">
                                                    <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 opacity-70 hover:opacity-80">
                                                        <circle cx="5" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <circle opacity="0.5" cx="12" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                        <circle cx="19" cy="12" r="2" stroke="currentColor" stroke-width="1.5"></circle>
                                                    </svg>
                                                </a>
                                                <ul x-cloak="" x-show="open" x-transition="" x-transition.duration.300ms="" class="ltr:right-0 rtl:left-0">
                                                    <li><a href="javascript:;" @click="toggle">View Report</a></li>
                                                    <li><a href="javascript:;" @click="toggle">Edit Report</a></li>
                                                </ul>
                                            </div>
                                        </div>
                                        <div class="relative mt-10">
                                            <div class="absolute -bottom-12 h-24 w-24 ltr:-right-12 rtl:-left-12">
                                                <svg width="24" height="24" viewbox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="h-full w-24 text-danger opacity-20">
                                                    <circle opacity="0.5" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="1.5"></circle>
                                                    <path d="M12 7V13" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                    <circle cx="12" cy="16" r="1" fill="currentColor"></circle>
                                                </svg>
                                            </div>

                                            <div class="grid grid-cols-2 gap-6 md:grid-cols-3">
                                                <div>
                                                    <div class="text-primary">Card Limit</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                                <div>
                                                    <div class="text-primary">Spent</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                                <div>
                                                    <div class="text-primary">Minimum</div>
                                                    <div class="mt-2 text-2xl font-semibold">N/A</div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Recent Transactions -->
                                <!-- Recent Transactions -->
                                <div class="panel">
                                    <div class="mb-5 text-lg font-bold">Recent Transactions</div>
                                    <div class="table-responsive">
                                        <table>
                                            <thead>
                                                <tr>
                                                    <th class="ltr:rounded-l-md rtl:rounded-r-md">ID</th>
                                                    <th>DATE</th>
                                                    <th>PROFILE</th>
                                                    <th>AMOUNT</th>
                                                    <th class="text-center ltr:rounded-r-md rtl:rounded-l-md">STATUS</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php
                                                $sql = "SELECT id, created_at, profile, amount, status FROM transaction_history WHERE status != '' ORDER BY created_at DESC LIMIT 6";
                                                $result = $conn->query($sql);
                                                if ($result && $result->num_rows > 0):
                                                    while ($row = $result->fetch_assoc()):
                                                        // Format date
                                                        $date = date('M d, Y', strtotime($row['created_at']));
                                                        // Format amount
                                                        $amount = 'GH¢ ' . number_format($row['amount'], 2);
                                                        // Status badge
                                                        $status = ucfirst($row['status']);
                                                        $badgeClass = 'bg-info/20 text-info';
                                                        if (strtolower($status) === 'completed') $badgeClass = 'bg-success/20 text-success';
                                                        elseif (strtolower($status) === 'pending') $badgeClass = 'bg-danger/20 text-danger';
                                                        ?>
                                                        <tr>
                                                            <td class="font-semibold">#<?= str_pad($row['id'], 2, '0', STR_PAD_LEFT) ?></td>
                                                            <td class="whitespace-nowrap"><?= $date ?></td>
                                                            <td class="whitespace-nowrap"><?= htmlspecialchars($row['profile']) ?></td>
                                                            <td><?= $amount ?></td>
                                                            <td class="text-center">
                                                                <span class="badge rounded-full <?= $badgeClass ?> hover:top-0"><?= $status ?></span>
                                                            </td>
                                                        </tr>
                                                    <?php endwhile;
                                                else: ?>
                                                    <tr>
                                                        <td colspan="5" class="text-center">No recent transactions found.</td>
                                                    </tr>
                                                <?php endif; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- start footer section -->
                <div class="p-6 pt-0 mt-auto text-center dark:text-white-dark ltr:sm:text-left rtl:sm:text-right">
                    © <span id="footer-year">2022</span>. Ewusiwa Hostel Wifi All rights reserved.
                </div>
                <!-- end footer section -->
            </div>
        </div>

        <script src="assets/js/alpine-collaspe.min.js"></script>
        <script src="assets/js/alpine-persist.min.js"></script>
        <script defer="" src="assets/js/alpine-ui.min.js"></script>
        <script defer="" src="assets/js/alpine-focus.min.js"></script>
        <script defer="" src="assets/js/alpine.min.js"></script>
        <script src="assets/js/custom.js"></script>
        <script defer="" src="assets/js/apexcharts.js"></script>
        <script>
            // main section
            document.addEventListener('alpine:init', () => {
                Alpine.data('scrollToTop', () => ({
                    showTopButton: false,
                    init() {
                        window.onscroll = () => {
                            this.scrollFunction();
                        };
                    },

                    scrollFunction() {
                        if (document.body.scrollTop > 50 || document.documentElement.scrollTop > 50) {
                            this.showTopButton = true;
                        } else {
                            this.showTopButton = false;
                        }
                    },

                    goToTop() {
                        document.body.scrollTop = 0;
                        document.documentElement.scrollTop = 0;
                    },
                }));

                // theme customization
                Alpine.data('customizer', () => ({
                    showCustomizer: false,
                }));

                // sidebar section
                Alpine.data('sidebar', () => ({
                    init() {
                        const selector = document.querySelector('.sidebar ul a[href="' + window.location.pathname + '"]');
                        if (selector) {
                            selector.classList.add('active');
                            const ul = selector.closest('ul.sub-menu');
                            if (ul) {
                                let ele = ul.closest('li.menu').querySelectorAll('.nav-link');
                                if (ele) {
                                    ele = ele[0];
                                    setTimeout(() => {
                                        ele.click();
                                    });
                                }
                            }
                        }
                    },
                }));

                // header section
                Alpine.data('header', () => ({
                    init() {
                        const selector = document.querySelector('ul.horizontal-menu a[href="' + window.location.pathname + '"]');
                        if (selector) {
                            selector.classList.add('active');
                            const ul = selector.closest('ul.sub-menu');
                            if (ul) {
                                let ele = ul.closest('li.menu').querySelectorAll('.nav-link');
                                if (ele) {
                                    ele = ele[0];
                                    setTimeout(() => {
                                        ele.classList.add('active');
                                    });
                                }
                            }
                        }
                    },

                    notifications: [
                        {
                            id: 1,
                            profile: 'user-profile.jpeg',
                            message: '<strong class="text-sm mr-1">Ewusiwa Hostel Wifi</strong>invite you to <strong>Prototyping</strong>',
                            time: '45 min ago',
                        },
                        {
                            id: 2,
                            profile: 'profile-34.jpeg',
                            message: '<strong class="text-sm mr-1">Adam Nolan</strong>mentioned you to <strong>UX Basics</strong>',
                            time: '9h Ago',
                        },
                        {
                            id: 3,
                            profile: 'profile-16.jpeg',
                            message: '<strong class="text-sm mr-1">Anna Morgan</strong>Upload a file',
                            time: '9h Ago',
                        },
                    ],

                    messages: [
                        {
                            id: 1,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-success-light dark:bg-success text-success dark:text-success-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path></svg></span>',
                            title: 'Congratulations!',
                            message: 'Your OS has been updated.',
                            time: '1hr',
                        },
                        {
                            id: 2,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-info-light dark:bg-info text-info dark:text-info-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><line x1="12" y1="16" x2="12" y2="12"></line><line x1="12" y1="8" x2="12.01" y2="8"></line></svg></span>',
                            title: 'Did you know?',
                            message: 'You can switch between artboards.',
                            time: '2hr',
                        },
                        {
                            id: 3,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-danger-light dark:bg-danger text-danger dark:text-danger-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg></span>',
                            title: 'Something went wrong!',
                            message: 'Send Reposrt',
                            time: '2days',
                        },
                        {
                            id: 4,
                            image: '<span class="grid place-content-center w-9 h-9 rounded-full bg-warning-light dark:bg-warning text-warning dark:text-warning-light"><svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">    <circle cx="12" cy="12" r="10"></circle>    <line x1="12" y1="8" x2="12" y2="12"></line>    <line x1="12" y1="16" x2="12.01" y2="16"></line></svg></span>',
                            title: 'Warning',
                            message: 'Your password strength is low.',
                            time: '5days',
                        },
                    ],

                    languages: [
                        {
                            id: 1,
                            key: 'Khmer',
                            value: 'kh',
                        },
                        {
                            id: 2,
                            key: 'Danish',
                            value: 'da',
                        },
                        {
                            id: 3,
                            key: 'English',
                            value: 'en',
                        },
                        {
                            id: 4,
                            key: 'French',
                            value: 'fr',
                        },
                        {
                            id: 5,
                            key: 'German',
                            value: 'de',
                        },
                        {
                            id: 6,
                            key: 'Greek',
                            value: 'el',
                        },
                        {
                            id: 7,
                            key: 'Hungarian',
                            value: 'hu',
                        },
                        {
                            id: 8,
                            key: 'Italian',
                            value: 'it',
                        },
                        {
                            id: 9,
                            key: 'Japanese',
                            value: 'ja',
                        },
                        {
                            id: 10,
                            key: 'Polish',
                            value: 'pl',
                        },
                        {
                            id: 11,
                            key: 'Portuguese',
                            value: 'pt',
                        },
                        {
                            id: 12,
                            key: 'Russian',
                            value: 'ru',
                        },
                        {
                            id: 13,
                            key: 'Spanish',
                            value: 'es',
                        },
                        {
                            id: 14,
                            key: 'Swedish',
                            value: 'sv',
                        },
                        {
                            id: 15,
                            key: 'Turkish',
                            value: 'tr',
                        },
                        {
                            id: 16,
                            key: 'Arabic',
                            value: 'ae',
                        },
                    ],

                    removeNotification(value) {
                        this.notifications = this.notifications.filter((d) => d.id !== value);
                    },

                    removeMessage(value) {
                        this.messages = this.messages.filter((d) => d.id !== value);
                    },
                }));

                // finance
                Alpine.data('finance', () => ({
                    init() {
                        const bitcoin = null;
                        const ethereum = null;
                        const litecoin = null;
                        const binance = null;
                        const tether = null;
                        const solana = null;

                        setTimeout(() => {
                            this.bitcoin = new ApexCharts(this.$refs.bitcoin, this.bitcoinOptions);
                            this.bitcoin.render();

                            this.ethereum = new ApexCharts(this.$refs.ethereum, this.ethereumOptions);
                            this.ethereum.render();

                            this.litecoin = new ApexCharts(this.$refs.litecoin, this.litecoinOptions);
                            this.litecoin.render();

                            this.binance = new ApexCharts(this.$refs.binance, this.binanceOptions);
                            this.binance.render();

                            this.tether = new ApexCharts(this.$refs.tether, this.tetherOptions);
                            this.tether.render();

                            this.solana = new ApexCharts(this.$refs.solana, this.solanaOptions);
                            this.solana.render();
                        }, 300);
                    },

                    get bitcoinOptions() {
                        return {
                            series: [
                                {
                                    data: [21, 9, 36, 12, 44, 25, 59, 41, 25, 66],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#00ab55'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },

                    get ethereumOptions() {
                        return {
                            series: [
                                {
                                    data: [44, 25, 59, 41, 66, 25, 21, 9, 36, 12],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#e7515a'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },

                    get litecoinOptions() {
                        return {
                            series: [
                                {
                                    data: [9, 21, 36, 12, 66, 25, 44, 25, 41, 59],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#00ab55'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },

                    get binanceOptions() {
                        return {
                            series: [
                                {
                                    data: [25, 44, 25, 59, 41, 21, 36, 12, 19, 9],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#e7515a'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },

                    get tetherOptions() {
                        return {
                            series: [
                                {
                                    data: [21, 59, 41, 44, 25, 66, 9, 36, 25, 12],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#00ab55'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },

                    get solanaOptions() {
                        return {
                            series: [
                                {
                                    data: [21, -9, 36, -12, 44, 25, 59, -41, 66, -25],
                                },
                            ],
                            chart: {
                                height: 45,
                                type: 'line',
                                sparkline: {
                                    enabled: true,
                                },
                            },
                            stroke: {
                                width: 2,
                            },
                            markers: {
                                size: 0,
                            },
                            colors: ['#e7515a'],
                            grid: {
                                padding: {
                                    top: 0,
                                    bottom: 0,
                                    left: 0,
                                },
                            },
                            tooltip: {
                                x: {
                                    show: false,
                                },
                                y: {
                                    title: {
                                        formatter: (formatter = () => {
                                            return '';
                                        }),
                                    },
                                },
                            },
                            responsive: [
                                {
                                    breakPoint: 576,
                                    options: {
                                        chart: {
                                            height: 95,
                                        },
                                        grid: {
                                            padding: {
                                                top: 45,
                                                bottom: 0,
                                                left: 0,
                                            },
                                        },
                                    },
                                },
                            ],
                        };
                    },
                }));
            });
        </script>
        <!-- <script>
document.addEventListener('alpine:init', () => {
    Alpine.data('finance', () => ({
        init() {
            <?php foreach ($topProfilesThisMonth as $i => $profile) {
                $monthlyRevenue = getProfileMonthlyRevenue($conn, $profile['profile']);
                $xRef = ['bitcoin', 'ethereum', 'litecoin'][$i];
                $color = ['#00ab55', '#e7515a', '#345d9d'][$i];
            ?>
            setTimeout(() => {
                new ApexCharts(this.$refs.<?= $xRef ?>, {
                    series: [{ data: <?= json_encode($monthlyRevenue) ?> }],
                    chart: { height: 45, type: 'line', sparkline: { enabled: true } },
                    stroke: { width: 2 },
                    markers: { size: 0 },
                    colors: ['<?= $color ?>'],
                    grid: { padding: { top: 0, bottom: 0, left: 0 } },
                    tooltip: {
                        x: { show: false },
                        y: { title: { formatter: () => '' } }
                    },
                    responsive: [{
                        breakPoint: 576,
                        options: {
                            chart: { height: 95 },
                            grid: { padding: { top: 45, bottom: 0, left: 0 } }
                        }
                    }]
                }).render();
            }, 300);
            <?php } ?>
        }
    }));
});
</script> -->
    </body>
</html>
