<?php
// Error reporting for debugging (remove in production)
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Fixed log file path - use absolute path
$logDir = '/var/www/logs/';
$logFile = $logDir . 'debug.log';

// Create directory if doesn't exist
if (!file_exists($logDir)) {
    mkdir($logDir, 0755, true);
}

function logMessage($message) {
    global $logFile;
    $status = file_put_contents($logFile, date('[Y-m-d H:i:s] ') . $message . PHP_EOL, FILE_APPEND);
    if ($status === false) {
        error_log("Failed to write to log file: " . $logFile);
    }
}

// Start logging
logMessage("===== Script Started =====");

// Verify we can write to log
if (!is_writable($logFile)) {
    $error = "Log file not writable: " . $logFile;
    error_log($error);
    die(json_encode(['status' => 'error', 'message' => $error]));
}

// Include database config
include 'database.php';
logMessage("Database config loaded");

// Verify database connection
try {
    $conn = new mysqli($host, $user, $pass, $db);
    if ($conn->connect_error) {
        throw new Exception("DB connection failed: " . $conn->connect_error);
    }
    logMessage("Database connected successfully");
} catch (Exception $e) {
    logMessage($e->getMessage());
    die(json_encode(['status' => 'error', 'message' => $e->getMessage()]));
}

// Get POST data
$raw = file_get_contents("php://input");
logMessage("Raw input: " . $raw);

if (empty($raw)) {
    logMessage("Error: Empty input received");
    die(json_encode(['status' => 'error', 'message' => 'No data received']));
}

$data = json_decode($raw, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    logMessage("JSON decode error: " . json_last_error_msg());
    die(json_encode(['status' => 'error', 'message' => 'Invalid JSON']));
}

// Verify required fields
$required = ['name', 'profile'];
foreach ($required as $field) {
    if (!isset($data[$field])) {
        logMessage("Missing required field: $field");
        die(json_encode(['status' => 'error', 'message' => "Missing $field"]));
    }
}

// Prepare statement
$sql = "INSERT INTO mikrotik_users (
    name, enabled, profile, limit_uptime, limit_bytes_in, limit_bytes_out,
    limit_bytes_total, uptime, bytes_in, packets_in, bytes_out, packets_out
) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    $error = "Prepare failed: " . $conn->error;
    logMessage($error);
    die(json_encode(['status' => 'error', 'message' => $error]));
}

// Bind parameters with null checks
$types = "sissiiiisiii";
$params = [
    $data['name'] ?? '',
    $data['enabled'] ?? 0,
    $data['profile'] ?? '',
    $data['limit_uptime'] ?? null,
    $data['limit_bytes_in'] ?? null,
    $data['limit_bytes_out'] ?? null,
    $data['limit_bytes_total'] ?? null,
    $data['uptime'] ?? null,
    $data['bytes_in'] ?? 0,
    $data['packets_in'] ?? 0,
    $data['bytes_out'] ?? 0,
    $data['packets_out'] ?? 0
];

logMessage("Parameters: " . print_r($params, true));

$stmt->bind_param($types, ...$params);

// Execute
if ($stmt->execute()) {
    $response = ['status' => 'success', 'id' => $conn->insert_id];
    logMessage("Insert successful. ID: " . $conn->insert_id);
} else {
    $error = "Execute failed: " . $stmt->error;
    logMessage($error);
    $response = ['status' => 'error', 'message' => $error];
}

$stmt->close();
$conn->close();

header('Content-Type: application/json');
echo json_encode($response);

logMessage("===== Script Completed =====" . PHP_EOL);
?>