<?php
header('Content-Type: application/json');
require_once __DIR__ . '/dashboard/database.php';

function log_debug($msg) {
    file_put_contents('upsert_hotspot_account_log.txt', date('Y-m-d H:i:s') . " - $msg\n", FILE_APPEND);
}

$data = json_decode(file_get_contents('php://input'), true);

$username = $data['username'] ?? null;
$password = $data['password'] ?? null;
$account_name = $data['account_name'] ?? null;
$enabled = $data['enabled'] ?? '1';
$profile = $data['profile'] ?? null;
$limit_uptime = $data['limit_uptime'] ?? null;
$limit_bytes_in = isset($data['limit_bytes_in']) ? $data['limit_bytes_in'] : null;
$limit_bytes_out = isset($data['limit_bytes_out']) ? $data['limit_bytes_out'] : null;
$limit_bytes_total = isset($data['limit_bytes_total']) ? $data['limit_bytes_total'] : null;
$transaction_id = $data['transaction_id'] ?? null;

if (!$username || !$password) {
    echo json_encode(['success' => false, 'error' => 'Username and password are required.']);
    exit;
}

$conn = getDBConnection();

try {
    $sql = "
        INSERT INTO hotspot_accounts 
            (username, password, account_name, enabled, profile, limit_uptime, limit_bytes_in, limit_bytes_out, limit_bytes_total, transaction_id)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
            password=VALUES(password),
            account_name=VALUES(account_name),
            enabled=VALUES(enabled),
            profile=VALUES(profile),
            limit_uptime=VALUES(limit_uptime),
            limit_bytes_in=VALUES(limit_bytes_in),
            limit_bytes_out=VALUES(limit_bytes_out),
            limit_bytes_total=VALUES(limit_bytes_total),
            transaction_id=VALUES(transaction_id)
    ";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        log_debug("Prepare failed: " . $conn->error);
        echo json_encode(['success' => false, 'error' => $conn->error]);
        exit;
    }
    $stmt->bind_param(
        'ssssssiiis',
        $username,
        $password,
        $account_name,
        $enabled,
        $profile,
        $limit_uptime,
        $limit_bytes_in,
        $limit_bytes_out,
        $limit_bytes_total,
        $transaction_id
    );
    $stmt->execute();
    $affected = $stmt->affected_rows;
    $stmt->close();
    log_debug("Upserted hotspot_accounts for $username. Affected rows: $affected");

    // --- NEW: Lookup account name from external service ---
    $lookup_url = "https://bundle.krolatech.org/korba_lookup.php";
    $lookup_payload = json_encode(['mobile_number' => $username]); // Use mobile_number as key
    $ch = curl_init($lookup_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $lookup_payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    $lookup_response = curl_exec($ch);
    $lookup_http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    log_debug("Lookup response ($lookup_http_code): $lookup_response");

    if ($lookup_http_code == 200 && $lookup_response) {
        $lookup_data = json_decode($lookup_response, true);
        if (is_array($lookup_data) && isset($lookup_data['mobile_number']) && isset($lookup_data['account_name'])) {
            $number = $lookup_data['mobile_number'];
            $account_name_from_lookup = $lookup_data['account_name'];

            // Update the account_name in the database
            $update_stmt = $conn->prepare("UPDATE hotspot_accounts SET account_name=? WHERE username=?");
            $update_stmt->bind_param('ss', $account_name_from_lookup, $number);
            $update_stmt->execute();
            $update_stmt->close();

            log_debug("Updated account_name for $number to $account_name_from_lookup");
        } else {
            log_debug("Lookup response missing expected fields.");
        }
    } else {
        log_debug("Failed to get valid response from lookup endpoint.");
    }

    echo json_encode(['success' => true, 'affected_rows' => $affected]);
} catch (Throwable $e) {
    log_debug("Exception during upsert: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
$conn->close();
?>